/**
 * \file: create_keys_cmd.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include "create_keys_cmd/create_keys_cmd.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <limits.h>

#include "control/configuration_defaults.h"
#include "model/daemon_model.h"

#include "util/helper.h"
#include "util/cmdline_parser.h"
#include "util/cfgfile_parser.h"


#include "create_signature_db_cmd/feature_db.h"
#include "create_signature_db_cmd/level_db.h"
#include "create_signature_db_cmd/signature_db_writer.h"

static cfg_item_t script_root_cfg	= {.spec = &script_root_cfg_spec, .cmdline = &script_root_cmdline_spec };
static cfg_item_t sig_db_key_cfg 	= {.spec = &sigdb_key_cfg_spec, .cmdline = &sigdb_key_cmdline_spec };
static cfg_item_t sysroot_cfg		= {.spec = &sysroot_cfg_spec, .cmdline = &sysroot_cmdline_spec };
static cfg_item_t priv_key_dest_cfg	= {.spec = &priv_key_dest_cfg_spec, .cmdline = &priv_key_dest_cmdline_spec };
static cfg_item_t conffile_cfg 		= {.spec = &conf_file_cfg_spec, .cmdline = &conffile_cmdline_spec };
static cfg_item_t help_cfg		= {.spec = &help_cfg_spec, .cmdline = &help_cmdline_spec };

static cfg_item_t *const create_key_cfgs[] = {
		&script_root_cfg,
		&sig_db_key_cfg,
		&sysroot_cfg,
		&priv_key_dest_cfg,
		&conffile_cfg,
		&help_cfg
};

static error_code_t create_keys_cmd_init(void);
static error_code_t create_keys_cmd_parse_args(const char *binary_name, int argc, char *argv[]);
static error_code_t create_keys_cmd_start(void);
static void create_keys_cmd_print_help(void);
static void create_keys_cmd_print_usage(void);
static error_code_t create_keys_cmd_create_level_keys(void);
static error_code_t create_keys_cmd_create_key(const char *priv_key_path, const char *pub_key_path);
static void create_keys_cmd_deinit(void);

static error_code_t create_keys_cmd_init(void)
{
	logger_init_console_only(LOGGER_LEVEL_ERROR);
	return RESULT_OK;
}

static error_code_t create_keys_cmd_parse_args(const char *binary_name, int argc, char *argv[])
{
	error_code_t result = RESULT_OK;
	int create_key_cfgs_cnt = ARRAY_SIZE(create_key_cfgs);

	(void) binary_name;

	helper_items_init(create_key_cfgs, create_key_cfgs_cnt);

	result = cmdline_parser_parse_cmd_line(create_key_cfgs, create_key_cfgs_cnt, argv, argc);

	if (result == RESULT_INVALID_ARGS)
	{
		create_keys_cmd_print_usage();
		return RESULT_INVALID_ARGS;
	}

	if (helper_get_flag(&help_cfg))
	{
		create_keys_cmd_print_help();
		return RESULT_HELP_PRINTED;
	}

	if (cfgfile_parser_is_any_cfg_item_unset(create_key_cfgs, create_key_cfgs_cnt))
		result = cfgfile_parser_parse_config_file(helper_get_str_value(&conffile_cfg),
				create_key_cfgs, create_key_cfgs_cnt);

	return result;
}

static void create_keys_cmd_print_help(void)
{
	printf("\n");
	printf("Authorization Level Daemon Initialization Utility - Used to create configurations and\n"
			"signatures for the ADIT authorization level daemon.\n\n");
	printf("Usage: ald_init %s [OPTIONS...]\n",create_keys_cmd_vtable.command);
	printf("This command creates private/public key pairs for generating and signing the signature db\n"
			"and for changing security levels\n\n");
	printf("\t"SCRIPT_ROOT_CMD_SHORT", "SCRIPT_ROOT_CMD_LONG"\t\tRoot dir of the script repository\n");
	printf("\t"SYSROOT_CMD_SHORT", "SYSROOT_CMD_LONG"\t\tDestination of an offline rfs to which the db is to be stored\n");
	printf("\t"SIGDB_KEY_CMD_SHORT", "SIGDB_KEY_CMD_LONG"\tPath to the file where the db pub key is stored in\n");
	printf("\t"PRIVKEY_DEST_CMD_SHORT", "PRIVKEY_DEST_CMD_LONG"\tDirectory to where the private keys are to be stored\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad sigdbpubkeypath and rootdir path from this config file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
	printf("The private keys are stored in the path set with "PRIVKEY_DEST_CMD_LONG". If the path\n"
			"is not passed, the keys are stored in the directory from which the command is called.\n\n"
			"The name and the destination of the signature db public key file can be set with the\n"
			""SIGDB_KEY_CMD_LONG" option. This name will be extended by \'"PUBLIC_KEY_EXT"\' while generating the keys.\n"
			"Please note that the signature db file itself does not need to be present when the keys\n"
			"are generated.If sigdbpubkey path is not passed, then its looked in the config file\n"
			"(default config file : \"%s\"). If sigdbpubkey path is not found in config file,\n"
			"then the default location \"%s\" is taken.\n\n"
			"The public keys for the level change requests are stored in the level configuration\n"
			"folders one for each level. The level configuration is searched below rootdir.\n"
			"Please refer to the manual document for the exact structure expected below rootdir.\n"
			"If the rootdir is not passed, its taken from the config file.If the rootdir is\n"
			"not available in conf file , the default directory \"%s\" is used\n\n"
			"A sysroot path can be passed using --sysroot option to create the database for an \n"
			"offline RFS on host side.\n\n", helper_get_str_default(conffile_cfg.spec),
			helper_get_str_default(sig_db_key_cfg.spec), helper_get_str_default(script_root_cfg.spec));
}

static void create_keys_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: ald_init %s [OPTIONS...]\n\n",create_keys_cmd_vtable.command);
	printf("\t"SCRIPT_ROOT_CMD_SHORT", "SCRIPT_ROOT_CMD_LONG"\t\tRoot dir of the script repository\n");
	printf("\t"SYSROOT_CMD_SHORT", "SYSROOT_CMD_LONG"\t\tDestination of an offline rfs to which the db is to be stored\n");
	printf("\t"SIGDB_KEY_CMD_SHORT", "SIGDB_KEY_CMD_LONG"\tPath to the file where the db pub key is stored in\n");
	printf("\t"PRIVKEY_DEST_CMD_SHORT", "PRIVKEY_DEST_CMD_LONG"\tDirectory to where the private keys are to be stored\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad sigdbpubkeypath and rootdir path from this config  file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
}

static error_code_t create_keys_cmd_start(void)
{
	error_code_t result = RESULT_OK;
	char pubkey_path[PATH_MAX];
	char privkey_path[PATH_MAX];

	printf("Generating key value pair for the signature db ...\n");
	printf("-------------------------------------------------------------------------------------\n");
	//creating the signature db keys
	snprintf(pubkey_path,PATH_MAX,"%s%s%s",helper_get_str_value(&sysroot_cfg),
			helper_get_str_value(&sig_db_key_cfg),PUBLIC_KEY_EXT);
	snprintf(privkey_path,PATH_MAX,"%s/%s",helper_get_str_value(&priv_key_dest_cfg),
			SIGDB_PRIV_KEY_NAME);
	result=create_keys_cmd_create_key(privkey_path,pubkey_path);
	printf("-------------------------------------------------------------------------------------\n");

	printf("Generating key value pair for the level changes ...\n\n");
	if (result==RESULT_OK)
		result=create_keys_cmd_create_level_keys();

	return result;
}

static error_code_t create_keys_cmd_create_level_keys(void)
{
	error_code_t result=RESULT_OK;
	DIR *dp;
	struct dirent *ep;
	char level_scripts_path[PATH_MAX];

	snprintf(level_scripts_path,PATH_MAX,"%s%s%s/",helper_get_str_value(&sysroot_cfg),
			helper_get_str_value(&script_root_cfg), LEVEL_SUBDIR);
	dp = opendir (level_scripts_path);
	if (dp == NULL)
	{
		printf("Unable to open the level configuration in \'%s%s\'- %s.\n",
				helper_get_str_value(&sysroot_cfg), helper_get_str_value(&script_root_cfg), strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		char pubkey_path[PATH_MAX];
		char privkey_path[PATH_MAX];

		security_level_t level;
		char *parse_result;

		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_DIR)
		{
			printf("WARNING: Unexpected element found in feature conf directory: \'%s\'. Ignoring it.\n",ep->d_name);
			continue;
		}

		level=(security_level_t)strtol(ep->d_name,&parse_result,10);
		if (level > MAX_LEVEL) {
			printf("ERROR: level %s is exceeding max allowed value.\n",ep->d_name);
			result=RESULT_INVALID_LEVEL;
			continue;
		}

		if (ep->d_name == parse_result || parse_result[0]!='\0')
		{
			printf("WARNING: Invalid level sub directory: %s. The name of a level sub directory"""
					" needs to be a positive number only. Ignoring the sub directory.\n",ep->d_name);
			continue;
		}

		snprintf(pubkey_path,PATH_MAX,"%s%d/%s%s",level_scripts_path, level, LEVEL_KEY_NAME, PUBLIC_KEY_EXT);
		snprintf(privkey_path,PATH_MAX,"%s/%s_%d%s",helper_get_str_value(&priv_key_dest_cfg),
				LEVEL_KEY_NAME, level, PRIVATE_KEY_EXT);
		printf("Security Level \'%d\' found. Creating a public/private key pair for the level ...\n",(int)level);
		printf("-------------------------------------------------------------------------------------\n");
		result=create_keys_cmd_create_key(privkey_path,pubkey_path);
		printf("-------------------------------------------------------------------------------------\n");
	}

	(void) closedir (dp);

	return result;
}

static error_code_t create_keys_cmd_create_key(const char *priv_key_path, const char *pub_key_path)
{
	char cmd[2*PATH_MAX];
	int exec_result;

	snprintf(cmd, 2*PATH_MAX,"/usr/bin/openssl genpkey -outform PEM -algorithm rsa -pkeyopt "
			"rsa_keygen_bits:%d -out %s",PRIV_KEY_LEN_BITS, priv_key_path);
	exec_result=system(cmd);
	if (exec_result==-1)
	{
		printf("Unable to call the \'/usr/bin/openssl\' command to create new key value pairs.\n");
		return RESULT_OPENSSL_MISSING;
	}

	if (exec_result!=0)
	{
		printf("The generation of the private key \'%s\' failed. Exit code of \'openssl\': %d.\n",
				priv_key_path, (((exec_result) & 0xff00) >> 8));
		return RESULT_OPENSSL_KEYGEN_FAILED;
	}

	snprintf(cmd, 2*PATH_MAX,"/usr/bin/openssl rsa -in %s -pubout -out %s",priv_key_path, pub_key_path);
	exec_result=system(cmd);

	if (exec_result==-1)
	{
		printf("Unable to call the \'/usr/bin/openssl\' command to create new key value pairs.\n");
		return RESULT_OPENSSL_MISSING;
	}

	if (exec_result!=0)
	{
		printf("The generation of the public key \'%s\' failed. Exit code of \'openssl\': %d.\n",
				pub_key_path, (((exec_result) & 0xff00) >> 8));
		return RESULT_OPENSSL_MISSING;
	}
	printf("\nPrivate key: %s\n",priv_key_path);
	printf("Public key: %s\n",pub_key_path);

	return RESULT_OK;
}

static void create_keys_cmd_deinit(void)
{
	helper_items_free(create_key_cfgs, ARRAY_SIZE(create_key_cfgs));
}

command_vtable_t create_keys_cmd_vtable=
{
		.command_description="Create the private/public key pairs for all levels and the signature data base.",
		.command="create_keys",
		.init=create_keys_cmd_init,
		.parse_args=create_keys_cmd_parse_args,
		.start=create_keys_cmd_start,
		.deinit=create_keys_cmd_deinit
};
